<?php

/**
 * HTPPG1 Email for Admin
 *
 * An email sent to the admin when an order status is changed to Pending Payment.
 * 
 * @class       HTPPG1_Email_Admin
 * @extends     WC_Email
 *
 */
class HTPPG1_Email_Admin extends WC_Email {

    function __construct() {

        // Add email ID, title, description, heading, subject
        $this->id = 'htppg1_email';
        $this->title = __('Howtopay Notification Email', 'htppg1-email');
        $this->description = __('This email is send by HowToPay Gateway for special notifications to both admin and customers', 'htppg1-email');

        $this->heading = __('HowToPay Gateway Notification', 'htppg1-email');
        $this->subject = __('[{blogname}] Attention required for a payment received to (Order #{order_number}) - {order_date}', 'htppg1-email');

        // email template path
        $this->template_html = 'emails/htppg1-item-email.php';
        $this->template_plain ='emails/plain/htppg1-item-email.php';
  
        // Triggers for this email
        //add_action('htppg1_pending_email_notification', array($this, 'trigger_notification_email'));
        //add_action('woocommerce_order_status_awaiting-confirm', array($this, 'trigger_notification_email'));
        //add_action('htppg1_item_email_notification', array($this, 'trigger'));

        // Call parent constructor
        parent::__construct();

        // Other settings
        $this->template_base = HTPPG1_TEMPLATE_PATH;
        
        // default the email recipient to the admin's email address
        $this->recipient = $this->get_option('recipient', get_option( 'admin_email' ));
        $this->order_id = "";
        $this->is_customer = false;
    }

    function replace_message_strings($sr_params , $string) {
        
        foreach($sr_params as $param) {
            $search = $param[0];
            $replace = $param[1];
            
            $string = str_replace($search, $replace, $string);
        }
        
        return $string;
    }
    
    public function trigger_notification_email($order_id, $post, $contexts, $customer=false) {

        $this->object = new WC_order($order_id);
        $this->order_id = $order_id;
        $this->is_customer = $customer;
        
        $currency_requested = isset($post['currency_requested']) ? $post['currency_requested']: "";
        $amount_requested = isset($post['amount_requested']) ? $post['amount_requested']: "";
        $currency_paid = isset($post['currency_paid']) ? $post['currency_paid']: "";
        $amount_paid = isset($post['amount_paid']) ? $post['amount_paid'] : "";
        $trn = isset($post['trn']) ? $post['trn'] : "";
        $invoice_no = isset($post['invoice_no']) ? $post['invoice_no'] : "";
                
        if($currency_requested == "") {
            $currency_requested = $this->object->get_currency();
        }

        if($amount_requested == "") {
            $amount_requested = $this->object->get_total();
        }
                
        $sr_params = array();
        $sr_params[] = array('{order_date}', date_i18n(wc_date_format(), strtotime($this->object->get_date_created())) );
        $sr_params[] = array('{received_date}', date_i18n(wc_date_format(), strtotime(date("Y-m-d H:i:s"))) );
        $sr_params[] = array('{requested_amount}', $currency_requested . " ". $amount_requested );
        $sr_params[] = array('{paid_amount}', $currency_paid . " ". $amount_paid );
        $sr_params[] = array('{invoice_no}', $invoice_no );
        $sr_params[] = array('{transaction_id}', $trn );
        $sr_params[] = array('{order_number}', $this->order_id );
        $sr_params[] = array('{context_topline}', $contexts['topline'] );
        $sr_params[] = array('{context_bottomline}', $contexts['bottomline'] );

        $this->send($this->get_recipient(), 
                $this->replace_message_strings($sr_params, $this->get_subject()), 
                $this->replace_message_strings($sr_params, $this->get_content()), $this->get_headers(), $this->get_attachments());
    }

    // return the html content
    function get_content_html() {
                
        ob_start();
        
        wc_get_template($this->template_html, array(
            'customer'=>$this->is_customer,
            'order' => $this->object,
            'email_heading' => $this->get_heading()
        ), 'htppg1-custom-email/', $this->template_base);
        
        return ob_get_clean();
    }

    // return the plain content
    function get_content_plain() {
        ob_start();
        
        wc_get_template($this->template_plain, array(
            'customer'=>$this->is_customer,
            'order' => $this->object,
            'email_heading' => $this->get_heading()
        ), 'htppg1-custom-email/', $this->template_base);
        
        return ob_get_clean();
    }

    // return the subject
    function get_subject() {

        //$order = new WC_order($this->order_id);
        return apply_filters('woocommerce_email_subject_' . $this->id, $this->format_string($this->subject), $this->object);
    }

    // return the email heading
    public function get_heading() {

        //$order = new WC_order($this->order_id);
        return apply_filters('woocommerce_email_heading_' . $this->id, $this->format_string($this->heading), $this->object);
    }

    // form fields that are displayed in WooCommerce->Settings->Emails
    function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Enable/Disable', 'htppg1-email'),
                'type' => 'checkbox',
                'label' => __('Enable this email notification', 'htppg1-email'),
                'default' => 'yes'
            ),
            'recipient' => array(
                'title' => __('Recipient', 'htppg1-email'),
                'type' => 'text',
                'description' => sprintf(__('Enter recipients (comma separated) for this email. Defaults to %s', 'htppg1-email'), get_option('admin_email')),
                'default' => get_option('admin_email')
            ),
            'subject' => array(
                'title' => __('Subject', 'htppg1-email'),
                'type' => 'text',
                'description' => sprintf(__('This controls the email subject line. Leave blank to use the default subject: <code>%s</code>.', 'htppg1-email'), $this->subject),
                'placeholder' => '',
                'default' => ''
            ),
            'heading' => array(
                'title' => __('Email Heading', 'htppg1-email'),
                'type' => 'text',
                'description' => sprintf(__('This controls the main heading contained within the email notification. Leave blank to use the default heading: <code>%s</code>.', 'htppg1-email'), $this->heading),
                'placeholder' => '',
                'default' => ''
            ),
            'email_type' => array(
                'title' => __('Email type', 'htppg1-email'),
                'type' => 'select',
                'description' => __('Choose which format of email to send.', 'htppg1-email'),
                'default' => 'html',
                'class' => 'email_type',
                'options' => array(
                    'plain' => __('Plain text', 'htppg1-email'),
                    'html' => __('HTML', 'htppg1-email'),
                    'multipart' => __('Multipart', 'htppg1-email'),
                )
            )
        );
    }

}

return new HTPPG1_Email_Admin();
