<?php
// htp_oc_plugin_v1
class HowtopayApi {
    private $api_key;
    private $secret_key;
    private $is_sandbox;
    private $api_base_url_live = 'https://api.howtopay.com/v2/';
    private $api_base_url_sandbox = 'https://api.howtopay.com/v2/'; // Please verify this sandbox URL

    public function __construct($api_key, $secret_key, $is_sandbox = false) {
        $this->api_key = $api_key;
        $this->secret_key = $secret_key;
        $this->is_sandbox = (bool)$is_sandbox;
    }

    private function getApiUrl() {
        return $this->is_sandbox ? $this->api_base_url_sandbox : $this->api_base_url_live;
    }

    private function getStringToSign($json_string) {
        return md5($json_string);
    }

    private function getSignature($post_data) {
        $json_string = json_encode($post_data);
        $string_to_sign = $this->getStringToSign($json_string);
        return hash_hmac('sha256', $string_to_sign, $this->secret_key);
    }

    public function request($action, $post_data) {
        $url = $this->getApiUrl() . $action;

        if($this->is_sandbox){
            $post_data['api_mode'] = 'stg_018'; //connects to stg18
        } else {
            $post_data['api_mode'] = 'live';
        }

        //  var_dump($action);
        //  var_dump($post_data); 
        //  exit;
        
        $signature = $this->getSignature($post_data);

        $headers = array(
            "cache-control: no-cache",
            "content-type: application/json",
            "htp-apiversion: HTPAPIv2",
            "htp-authorization: HTPAPIv2Auth " . $this->api_key . ":" . $signature,
            "htp-timestamp: " . time()
        );

        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode($post_data),
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ));

        $response_body = curl_exec($curl);
        $curl_error_number = curl_errno($curl);
        $curl_error_message = curl_error($curl);
        curl_close($curl);

        // var_dump($post_data); 
        // var_dump($response_body); 
        // var_dump($curl_error_number); 
        // var_dump($curl_error_message);
        // exit;

        if ($curl_error_number) {
            // You might want to log this error for debugging
            // error_log("HowToPay API cURL Error ($curl_error_number): $curl_error_message for URL: $url with payload: " . json_encode($post_data));
            return array(
                'type' => "error",
                "short_message" => 'curl_error',
                "message" => "cURL Error ({$curl_error_number}): {$curl_error_message}",
                "data" => array()
            );
        }

        $response_decoded = json_decode($response_body, true);

        // var_dump($response_body); 
        // var_dump($response_decoded); 
        // var_dump(json_last_error()); 
        // exit;

        if (json_last_error() !== JSON_ERROR_NONE) {
             // You might want to log this error for debugging
            // error_log("HowToPay API JSON Decode Error: " . json_last_error_msg() . " for response: " . $response_body);
            return array(
                'type' => "error",
                "short_message" => 'json_decode_error',
                "message" => "Failed to decode API response: " . json_last_error_msg(),
                "data" => array('raw_response' => $response_body)
            );
        }

        if ((isset($response_decoded['type']) && $response_decoded['type'] == 'error') || 
            (isset($response_decoded["message"]) && strtolower($response_decoded["message"]) == 'server error')) {
            return array(
                'type' => "error",
                "short_message" => (isset($response_decoded['short_message']) ? trim($response_decoded['short_message']) : 'api_error'),
                "message" => (isset($response_decoded['message']) ? trim($response_decoded['message']) : 'API Error'),
                "data" => isset($response_decoded['data']) ? $response_decoded['data'] : array()
            );
        } elseif (isset($response_decoded['type']) && $response_decoded['type'] == 'success') {
            return array(
                'type' => "success",
                "data" => isset($response_decoded['data']) ? $response_decoded['data'] : array()
            );
        } else {
            // error_log("HowToPay API Unexpected Response: " . $response_body);
            return array(
                'type' => "error",
                "short_message" => 'unexpected_response_structure',
                "message" => 'Unexpected API response structure.',
                "data" => $response_decoded
            );
        }
    }

    /**
     * Verifies the signature of an incoming webhook payload.
     * @param string $received_signature The signature from the 'htp-signature' header or payload.
     * @param array $data_payload The 'data' part of the webhook payload.
     * @return bool True if the signature is valid, false otherwise.
     */
    public function verifySignature($received_signature, $data_payload) {
        $expected_signature = $this->getSignature($data_payload); // Uses the existing getSignature method
        return hash_equals((string)$expected_signature, (string)$received_signature);
    }

    /**
     * Checks the connection with the HowToPay API server.
     * @return array API response
     */
    public function checkConnection() {
        $action = "gateway/check"; // API endpoint for checking connection
        $post_data = array('action' => 'check_connection'); // Payload for the check
        return $this->request($action, $post_data);
    }
}

?>
